
/*************************************************************************
**
** PROJECT:			dunsel app
**
** FILE:			App.c
**
***********************************************************************/

#define __MAIN__
#include <App.h>

#include <FeatureMgr.h>			// Needed to get the ROM version
#include <SoundMgr.h>


//- forms used from top level
#include "ListView.h"			// midi record list form
#include "EditView.h"			// midi record edit form



/***********************************************************************
 * Prototypes for internal functions
 **********************************************************************/
static Int ApplicationInit(void);
static void ApplicationCleanup(void);								// clean up before app exit
static Boolean ApplicationHandleEvent(EventPtr event);			// handle form load events
static void EventLoop(void);


// returns object ID of hit button
Word DoModalForm(Word formID)
{
    FormPtr previousForm = FrmGetActiveForm();
    FormPtr frm = FrmInitForm(formID);
    Word    hitButton;
    
    FrmSetActiveForm(frm);
    
// Set an event handler, if you wish, with FrmSetEventHandler
// Initialize any form objects in the form

    hitButton = FrmDoDialog(frm);
    
    // read any values from the form objects here
    // before the form is deleted
    
    if (previousForm)
        FrmSetActiveForm(previousForm);
        
    FrmDeleteForm(frm);
    return hitButton;
}

void DoAboutBox(void) 
{
	//	FrmAlert(AboutBoxAlert);
	VoidHand hAIN, hver; // hAIB, hInfo;
	CharPtr pAppName = "(appname)";
	CharPtr pAppVersion = "(appversion)";
	
	//! should get app info from resources
	//-	ainRsc(tAIN) = app name ID=ainID(1000)
	hAIN = DmGet1Resource(ainRsc, ainID);
	if (hAIN)
		pAppName = MemHandleLock(hAIN);
	
	//-	verRsc(tver) = version string (ID=appVersionID(1) or appVersionAlternateID(1000))
	//	(GCC tends to use appVersionID, CW uses appVersionAlternateID)
	hver = DmGet1Resource(verRsc, appVersionAlternateID);	// I use CW, check this first
	if (!hver)
		hver = DmGet1Resource(verRsc, appVersionID);
	if (hver)
		pAppVersion = MemHandleLock(hver);
	
#if 0
	//- appInfoStringsRsc(tAIS) = app info strings
	hInfo = DmGet1Resource(appInfoStringsRsc, 1000);
	if (hInfo)
		pAppInfo = MemHandleLock(hInfo);
#endif

	//- show it all now!
	FrmCustomAlert(AboutBoxAlert, pAppName, pAppVersion, "");
	
		
		
	//!	Copy code from DoModalForm(AboutBoxForm) here so we can add
	//	painting of the icon
	
#if 0
	//-	iconType(tAIB) = icon (22x22) ID=1000, mini icon (9x15) ID=1001
	// (constructor has something which is supposed to specify 
	// app ID, but launcher code I've seen assumes 1000)
	// System Default app icons (for apps missing a tAIB of their own):
	//  defaultAppIconBitmap(10000), defaultAppSmallIconBitmap(10001)
	hAIB = DmGet1Resource( (ULong)'tAIB', 1000 );
	if (hRes) {
		BytePtr pAIB = MemHandleLock( hAIB );
		WinDrawBitmap( (BitmapPtr) pAIB, xpos, ypos);
		MemHandleUnlock(hAIB);
	}
#endif

	//- clean up
	if (hAIN) MemHandleUnlock(hAIN);
	if (hver) MemHandleUnlock(hver);
//	if (hInfo) MemHandleUnlock(hInfo);
}

void callMenu(Word menuID) 
{
	EventType event;
    MemSet(&event, sizeof(EventType), 0);
	event.eType = menuEvent;
	event.data.menu.itemID = menuID;
	EvtAddEventToQueue(&event);
}

void callButton(Word ctlID) 
{
	EventType event;
    MemSet(&event, sizeof(EventType), 0);
	event.eType = ctlEnterEvent;
	event.data.ctlEnter.controlID = ctlID;
	EvtAddEventToQueue(&event);
}


// returns field that has the focus, if any, including in embedded tables
FieldPtr GetFocusObjectPtr (void)
{
   FormPtr frm;
   Word focus;
   FormObjectKind objType;
   
   frm = FrmGetActiveForm ();
   focus = FrmGetFocus (frm);
   if (focus == noFocus)
      return (NULL);
      
   objType = FrmGetObjectType (frm, focus);
   
   if (objType == frmFieldObj)
      return (FrmGetObjectPtr (frm, focus));
   
   else if (objType == frmTableObj)
      return (TblGetCurrentField (FrmGetObjectPtr (frm, focus)));
   
   return NULL;
}


Boolean HandleCommonMenuItems(Word menuID)
{
	FieldPtr   fld;
	
	Boolean handled = true;

	switch (menuID) {
	case EditUndo:
	case EditCut:
	case EditCopy:
	case EditPaste:
	case EditSelectAll:
		fld = GetFocusObjectPtr();
		if (!fld) 
			return false;
		if (menuID == EditUndo)
			FldUndo(fld);
		else if (menuID == EditCut)
			FldCut(fld);
		else if (menuID == EditCopy)
			FldCopy(fld);
		else if (menuID == EditPaste)
			FldCopy(fld);
		else if (menuID == EditSelectAll)
			FldSetSelection (fld, 0, FldGetTextLength (fld));
		break;

#if 0
	case EditKeyboard:
		SysKeyboardDialog(kbdDefault);
		break;

	case EditGrafitti: 
		SysGraffitiReferenceDialog(referenceDefault);
		break;
#endif

	case OptionsAbout:
		DoAboutBox();
		break;

	default:
		handled = false;;
	}
	
	return handled;
}


/***********************************************************************
 *
 * FUNCTION:     ApplicationInit
 *
 * DESCRIPTION:  This routine sets up the initial state of the application.
 *
 * PARAMETERS:   None.
 *
 * RETURNED:     0 = success; non-zero = error code
 *
 ***********************************************************************/
static Int ApplicationInit(void)
{
	// find out what version of the OS we're running on
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &gRomVersion);
	
	// Set current view so the first form to display is the main form.
	CurrentView = ListViewForm;


	return 0;
}


/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: Check that the ROM version meets your
 *              minimum requirement.  Warn if the app was switched to.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags indicating how the application was
 *											 launched.  A warning is displayed only if
 *                                these flags indicate that the app is 
 *											 launched normally.
 *
 * RETURNED:    zero if rom is compatible else an error code
 *                             
 ***********************************************************************/
static Err RomVersionCompatible (DWord requiredVersion, Word launchFlags)
{
	DWord romVersion;
	
	// See if we're on in minimum required version of the ROM or later.
	// The system records the version number in a feature.  A feature is a
	// piece of information which can be looked up by a creator and feature
	// number.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion)
		{
		// If the user launched the app from the launcher, explain
		// why the app shouldn't run.  If the app was contacted for something
		// else, like it was asked to find a string by the system find, then
		// don't bother the user with a warning dialog.  These flags tell how
		// the app was launched to decided if a warning should be displayed.
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
			{
			FrmAlert (RomIncompatibleAlert);
		
			// Pilot 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.  The sysFileCDefaultApp is considered "safe".
			if (romVersion < 0x02000000)
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
			}
		
		return (sysErrRomIncompatible);
		}

	return 0;
}

/***********************************************************************
 *
 * FUNCTION:    ApplicationCleanup
 *
 * DESCRIPTION: This routine closes the application's database
 *              and saves the current state of the application.
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 ***********************************************************************/
static void ApplicationCleanup(void)
{
	// Close all open forms to allow their frmCloseEvent handlers
	// to execute.  An appStopEvent doesn't send frmCloseEvents.
	// FrmCloseAllForms will send all opened forms a frmCloseEvent.
	FrmCloseAllForms ();
}

static Boolean ApplicationLoadForm(Int formId)
{
	FormPtr	frm;
	Boolean handled = true;

	// Load the form resource specified in the event then activate the form.
	frm = FrmInitForm(formId);	// can this fail?
	FrmSetActiveForm(frm);

	// Set the event handler for the form.  The handler of the currently 
	// active form is called by FrmDispatchEvent each time it receives an event.
	switch (formId)
		{
		case ListViewForm:
			FrmSetEventHandler(frm, ListViewEventHandler);
			break;
			
		case EditViewForm:
			FrmSetEventHandler(frm, EditViewEventHandler);
			break;
			
		default:
			handled = false;
			break;
		}
	
	return handled;
}


/***********************************************************************
 *
 * FUNCTION:    ApplicationHandleEvent
 *
 * DESCRIPTION: Loads a form resource and sets the event handler for the form.
 *
 * PARAMETERS:  event - a pointer to an EventType structure
 *
 * RETURNED:    True if the event has been handled and should not be
 *						passed to a higher level handler.
 *
 ***********************************************************************/
static Boolean ApplicationHandleEvent(EventPtr event)
{
	Boolean	handled = true;

	switch (event->eType) {
		case frmLoadEvent:
			handled = ApplicationLoadForm(event->data.frmLoad.formID);
			break;
			
		default:
			handled = false;
			break;
	}
	
	return handled;
}


/***********************************************************************
 *
 * FUNCTION:		EventLoop
 *
 * DESCRIPTION:	A simple loop that obtains events from the Event
 *						Manager and passes them on to various applications and
 *						system event handlers before passing them on to
 *						FrmDispatchEvent for default processing.
 *
 * PARAMETERS:		None.
 *
 * RETURNED:		Nothing.
 *
 ***********************************************************************/
static void EventLoop(void)
{
	EventType	event;
	Word			error;
	
	do
		{
		// Get the next available event.
		EvtGetEvent(&event, evtWaitForever);
		
		// Give the system a chance to handle the event.
		if (! SysHandleEvent(&event))

			// Give the menu bar a chance to update and handle the event.	
			if (! MenuHandleEvent(0, &event, &error))

				// Give the application a chance to handle the event.
				if (! ApplicationHandleEvent(&event))

					// Let the form object provide default handling of the event.
					FrmDispatchEvent(&event);
		}
	while (event.eType != appStopEvent);
}


/***********************************************************************
 *
 * FUNCTION:		PilotMain
 *
 * DESCRIPTION:	This function is the equivalent of a main() function
 *						under standard C.  It is called by the Emulator to begin
 *						execution of this application.
 *
 * PARAMETERS:		cmd - command specifying how to launch the application.
 *						cmdPBP - parameter block for the command.
 *						launchFlags - flags used to configure the launch.			
 *
 * RETURNED:		Any applicable error codes.
 *
 ***********************************************************************/
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
	Word error;		// error starting the app
	
	// This app makes use of PalmOS 2.0 features.  It will crash if
	// run on an earlier version of PalmOS.  Detect and warn if this happens,
	// then exit.
	// (There are other checks for 3.0 inside the code; it'll work somewhat on a 2.0 device.)
	error = RomVersionCompatible (version20, launchFlags);
	if (error) return error;

	// We only handle a normal launch... all other messages are ignored.
	if (cmd == sysAppLaunchCmdNormalLaunch) {
		// Initialize the application's global variables and database.
		if (ApplicationInit() == 0) {
			// Start the first form.
			FrmGotoForm(CurrentView);
				
			// Start the event loop.
			EventLoop();
			
		} else {
			// error initializing app, report it
			FrmCustomAlert (DBOpenFailureAlert, "System Midi Sounds","","");
		}
		
		// Clean up before exiting the applcation.
		ApplicationCleanup();
	}
		
	return 0;
}
